package cz.drg.clasificator;

import cz.drg.clasificator.util.UnmarshallOptimized;
import cz.drg.clasificator.exception.ShutdownException;
import cz.drg.clasificator.readers.InputReader;
import cz.drg.clasificator.util.Constants;
import cz.drg.clasificator.util.HeaderList;
import cz.drg.clasificator.util.OutputHelper;
import cz.drg.clasificator.writers.OutputWriter;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.IntStream;
import org.dmg.pmml.PMML;
import org.jpmml.evaluator.Evaluator;
import org.jpmml.evaluator.FieldValue;
import org.jpmml.evaluator.InputField;
import org.jpmml.evaluator.InvalidResultException;
import org.jpmml.evaluator.ModelEvaluatorFactory;
import org.jpmml.evaluator.OutputField;
import static cz.drg.clasificator.util.OutputHelper.*;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.nio.file.Files;
import java.nio.file.attribute.BasicFileAttributes;
import java.nio.file.attribute.FileTime;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.zip.DeflaterOutputStream;
import java.util.zip.InflaterInputStream;
import java.util.zip.ZipFile;
import org.jpmml.evaluator.EvaluationException;
import org.jpmml.evaluator.EvaluatorBuilder;
import org.jpmml.evaluator.ModelEvaluatorBuilder;
import org.jpmml.evaluator.ValueCheckException;
import org.jpmml.evaluator.visitors.DefaultModelEvaluatorBattery;
import org.jpmml.model.InvalidElementException;


/**
 * Application main purpose is to classify provided input data based on some PMML rules.
 * This class does exactly that. It expects InputReader/OutputWriter as a input and with 
 * that it reads, evaluates and writes the data.
 * 
 * @author Pavel Jirasek
 */
public class CZDRGClasificator {

    private PMML pmml;
    private Evaluator evaluator;
    private Map<String, Object> errorCodeMapping;

    public CZDRGClasificator() {
        this.errorCodeMapping = new HashMap<>();
        initErrorMapping();
    }
    
    /**
     * Initializes the PMML and its evaluator instance so its not needed to be created
     * over and over again with each call on {@link #startEvaluation(cz.drg.clasificator.readers.InputReader, cz.drg.clasificator.writers.OutputWriter)} method.
     * @param reader 
     */
    public void processPMML(InputReader reader){
        
        long now = System.currentTimeMillis();
            
        try {
        
            ZipFile pmmlZipFile = reader.readPmmlInput();
            File pmmlByteFile = new File(pmmlZipFile.getName().split("\\.")[0]+".model");
            
            if(!pmmlByteFile.exists()){
                dualLog("Missing PMML model file detected. Generating new PMML model.");
                unmarshalAndCreatePmmlByteModel(pmmlZipFile, pmmlByteFile);
                
                updatePmmlModelByteFileLastModifiedTime(pmmlZipFile, pmmlByteFile);
            }
            else{
                
                if(!getLastModifiedTime(pmmlZipFile).equals(getLastModifiedTime(pmmlByteFile))){
                    dualLog("Changed PMML zip file detected. Generating new PMML model.");
                    pmmlByteFile.delete();
                    
                    unmarshalAndCreatePmmlByteModel(pmmlZipFile, pmmlByteFile);
                    updatePmmlModelByteFileLastModifiedTime(pmmlZipFile, pmmlByteFile);
                }
                else{
                    try(ObjectInputStream ois2 = new ObjectInputStream(new InflaterInputStream(new FileInputStream(pmmlByteFile)))){
                        pmml = (PMML) ois2.readObject();
                    }
                }
                
            }
            
        } catch (IOException ex) {
            Logger.getLogger(CZDRGClasificator.class.getName()).log(Level.SEVERE, null, ex);
        } catch (ClassNotFoundException ex) {
            Logger.getLogger(CZDRGClasificator.class.getName()).log(Level.SEVERE, null, ex);
        }

        System.out.println("Unmarshaled PMML after "+(System.currentTimeMillis()-now)+"ms.");
        
        EvaluatorBuilder evaluatorBuilder = new ModelEvaluatorBuilder(pmml)
                .setModelEvaluatorFactory(ModelEvaluatorFactory.newInstance());

        evaluator = evaluatorBuilder.build();

        // Perform self-testing
        evaluator.verify();
    }
    

    private void unmarshalAndCreatePmmlByteModel(ZipFile pmmlZipFile, File pmmlByteFile) throws FileNotFoundException, IOException{
        pmml = new UnmarshallOptimized().unmarshal(pmmlZipFile);
        optimizePmml(pmml);

        try(ObjectOutputStream  os = new ObjectOutputStream (new DeflaterOutputStream (new FileOutputStream(pmmlByteFile)))){
            os.writeObject(pmml);
        }
    }
    
    /**
     * Only entry point into this class. This method starts the evaluation of data
     * provided by the InputReader and writes them with provided OutputWriter.
     * 
     * @param reader reader of input data
     * @param writer writer of evaluated data
     * @throws Exception 
     */
    public void startEvaluation(InputReader reader, OutputWriter writer) {

        long evaluationStartTime = System.currentTimeMillis();

        if(pmml == null || evaluator == null){
            processPMML(reader);
        }

        writer.clear();
        
        long readLines = 0;
        long writtenLines = 0;
        long errorLines = 0;
        long noClasificationLines = 0;
        
        while(reader.hasNextEntryBatch()){
            
            HeaderList inputList = reader.readNextEntryBatch();
            
            List<Map<String, ?>> outputRecords = evaluateInput(evaluator, inputList);
            List<List<String>> lines = transformOutput(evaluator, outputRecords);

            writer.writeOutput(inputList, lines);
            
            readLines += inputList.numOfLines();
            writtenLines += lines.size()-1;
            
            dualLog("Batch of size "+inputList.numOfLines());
            dualLog("Total of "+readLines+" entries.");
            
            for (Map<String, ?> outputRecord : outputRecords) {
                Object errorCode = outputRecord.get("CHYBA");
                Object clasification = outputRecord.get("DRG");
                
                if(outputRecord.isEmpty() || !errorCode.equals(0)){
                    errorLines++;
                }
                if(!outputRecord.isEmpty() && clasification.equals("missing")){
                    noClasificationLines++;
                }
            }
        }
        
        writer.close();
        reader.close();
        
        logResult(evaluationStartTime, readLines, writtenLines, errorLines, noClasificationLines);
        
        if(readLines != writtenLines){
            throw new ShutdownException(Constants.ERR_INCORRECT_RESULT_TRANSFORMATION);
        }
    }
    
    protected void logResult(long evaluationStartTime, long readLines, long writtenLines, long errorLines, long noClasificationLines){
        
        double timeElapsedSec = (System.currentTimeMillis() - evaluationStartTime) / 1000.0;
        
        separatorLog();   
        
        dualLog(String.format("%-16s%8d","Input entries",readLines).replaceAll("\\s(?=\\s+)", "."));
        dualLog(String.format("%-16s%8d","Output entries",writtenLines).replaceAll("\\s(?=\\s+)", "."));
        dualLog(String.format("%-16s%8d","Valid entries",(writtenLines-errorLines)).replaceAll("\\s(?=\\s+)", "."));
        dualLog(String.format("%-16s%8d","Error entries",errorLines).replaceAll("\\s(?=\\s+)", "."));
        dualLog(String.format("%-16s%8d","No clasification",noClasificationLines).replaceAll("\\s(?=\\s+)", "."));

        separatorLog();   

        dualLog(String.format("Evaluated after %.3f sec", timeElapsedSec));
        separatorLog();
    }
    

    /**
     * This method evaluates all input data based on provided PMML instance.
     * 
     * @param inputList all read input data
     * @param pmml PMML instance used for evaluation
     * @param modelEvaluatorFactory
     * @return 
     */
    private List<Map<String, ?>> evaluateInput(Evaluator evaluator2, HeaderList inputList) {

        List<Map<String, ?>> outputRecords = new ArrayList<>();
        fillWithNulls(inputList.numOfLines(), outputRecords);

        List<Map<String, String>> rows = inputList.getRows();
        IntStream.range(0, rows.size()).parallel().forEach(index -> {
            
            try{
                Map<String, ?> evaluated = prepareEagerlyAndEvaluate(evaluator2, rows.get(index));

                Map<String, Object> result = new HashMap<>();

                for (Map.Entry<String, ? extends Object> e : evaluated.entrySet()) {
                    result.put(e.getKey(), e.getValue());
                }

                outputRecords.set(index, result);
            }
            catch(EvaluationException ex){
                OutputHelper.dualLog("Something went wrong during evaluation.");
                OutputHelper.dualLog(ex.getMessage());
                System.exit(-1);
            }
            catch(InvalidElementException ex){
                OutputHelper.dualLog("Invalid element in model detected. Skipping entry triggering exception on index "+index+" of current batch.");
                OutputHelper.dualLog(ex.getMessage());
                Map<String, Object> result = new HashMap<>();
                outputRecords.set(index, result);
            }

        });

        return outputRecords;
    }
    
    private void initErrorMapping(){
        
        errorCodeMapping.put("LOS", 2);
        errorCodeMapping.put("VEKLET", 3);
        errorCodeMapping.put("VEKDEN", 3);
        errorCodeMapping.put("POHLAVI", 4);
        errorCodeMapping.put("UKONCENI", 5);
        errorCodeMapping.put("HMOTNOST", 6);
        errorCodeMapping.put("ODB_PRI", 7);
        errorCodeMapping.put("DG_HLAVNI", 8);
        errorCodeMapping.put("UPV", 9);
        errorCodeMapping.put("GEST_VEK", 10);
        errorCodeMapping.put("VERZE_P", 11);
        errorCodeMapping.put("OZ_DNY", 13);
        errorCodeMapping.put("RHB_DNY", 13);
        errorCodeMapping.put("PS_DNY", 13);
        errorCodeMapping.put("KRN_DNY", 13);
        errorCodeMapping.put("POP_DNY", 13);
        errorCodeMapping.put("DIA_DNY", 13);
        errorCodeMapping.put("HRU_DNY", 13);
        errorCodeMapping.put("BRI_DNY", 13);
        errorCodeMapping.put("ZLU_DNY", 13);
        errorCodeMapping.put("HRD_DNY", 13);
        errorCodeMapping.put("OKO_DNY", 13);
        errorCodeMapping.put("SRD_DNY", 13);
        errorCodeMapping.put("CEV_DNY", 13);
        errorCodeMapping.put("HDL_DNY", 13);
        errorCodeMapping.put("KP1", 14);
        errorCodeMapping.put("KP2", 14);
        errorCodeMapping.put("KP3", 14);
        errorCodeMapping.put("KP4", 14);
        errorCodeMapping.put("KP5", 14);
        errorCodeMapping.put("KP6", 14);
        errorCodeMapping.put("KP7", 14);
        errorCodeMapping.put("KP8", 14);
        errorCodeMapping.put("KP9", 14);
        errorCodeMapping.put("KP10", 14);
    }
    
            
            
    /**
     * This method evaluates single row of data. First it transform application data form
     * into the PMML evaluation library data form and than evaluates them.
     * 
     * @param evaluator PMML library evaluator
     * @param userArguments single row of input data
     * @return 
     */
    private Map<String, ?> prepareEagerlyAndEvaluate(Evaluator evaluator, Map<String, String> userArguments) {
        Map<String, FieldValue> pmmlArguments = new LinkedHashMap<>();

        List<InputField> activeFields = evaluator.getActiveFields();

        for (InputField activeField : activeFields) {
            // The key type of the user arguments map is java.lang.String.
            // A FieldName can be "unwrapped" to a String using FieldName#getValue().
            String userValue = userArguments.get(activeField.getFieldName());

            // The value type of the user arguments map is unknown.
            // An Object is converted to a String using Object#toString().
            // A missing value is represented by null.
//            if(activeField.getName().getValue().equalsIgnoreCase("krit_vyk_poc1")){
//                System.out.println("Field: "+activeField.getName().getValue()+" value: '"+userValue+"'");
//            }
            
            FieldValue pmmlValue = null;
            
            try{
                pmmlValue = activeField.prepare((userValue != null && !userValue.isEmpty() ? userValue : null));
            }
            catch(InvalidResultException | ValueCheckException ex){
                
                dualLog(ex.getMessage());
                
                String message = ex.getMessage();
                
                String errorFieldName = message.split("\"")[1];
                
                List<OutputField> outputFields = evaluator.getOutputFields();
                Map<String, Object> errorOutputRecord = new HashMap<>();
                
                for (OutputField outputField : outputFields) {
                    
                    if(outputField.getFieldName().equalsIgnoreCase("CHYBA")){
                        
                        errorOutputRecord.put(outputField.getName(), errorCodeMapping.getOrDefault(errorFieldName, -1));
                    }
                    else if(outputField.getFieldName().equalsIgnoreCase("verze_g")){
                        errorOutputRecord.put(outputField.getName(), getClass().getPackage().getImplementationVersion());
                    }
                    else if(outputField.getFieldName().equalsIgnoreCase("skore_zav")){
                        errorOutputRecord.put(outputField.getName(), 0);
                    }
                    else if(outputField.getFieldName().equalsIgnoreCase("DRG")){
                        
                        errorOutputRecord.put(outputField.getName(), "missing");
                    }
                    else if(outputField.getFieldName().equalsIgnoreCase("DRG_KAT")){
                        
                        errorOutputRecord.put(outputField.getName(), "missing");
                    }
                    else{
                        errorOutputRecord.put(outputField.getName(), "");
                    }
                    
                }
                return errorOutputRecord;
            }

            pmmlArguments.put(activeField.getName(), pmmlValue);
        }

        return evaluator.evaluate(pmmlArguments);
    }

    private void optimizePmml(PMML pmml) {
        DefaultModelEvaluatorBattery dmeb = new DefaultModelEvaluatorBattery();
        dmeb.applyTo(pmml);
    }

    private List<List<String>> transformOutput(Evaluator evaluator, List<Map<String, ?>> outputRecords) {

        List<OutputField> outputFields = evaluator.getOutputFields();

        List<List<String>> lines = new ArrayList<>();

        List<String> headerLine = new ArrayList();
        for (OutputField outputField : outputFields) {
            //skip output fields designed just for transformations in other output fields
            if(!outputField.isFinalResult()){
                continue;
            }
            
            headerLine.add(outputField.getFieldName());
        }
        lines.add(headerLine);

        List<String> line = new ArrayList();

        for (Map<String, ?> outputRecord : outputRecords) {

            for (OutputField outputField : outputFields) {
                //skip output fields designed just for transformations in other output fields
                if(!outputField.isFinalResult()){
                    continue;
                }
                
                if (outputRecord.containsKey(outputField.getFieldName())) {
                    line.add(String.valueOf(outputRecord.get(outputField.getFieldName())));
                }

            }

            lines.add(line);
            line = new ArrayList<>();

        }

        return lines;

    }
    
    private void updatePmmlModelByteFileLastModifiedTime(ZipFile pmmlZipFile, File pmmlByteFile) throws IOException{
        Files.setLastModifiedTime(pmmlByteFile.toPath(), getLastModifiedTime(pmmlZipFile));
    }
    
    private FileTime getLastModifiedTime(ZipFile file) throws IOException{
        return getLastModifiedTime(new File(file.getName()));
    }
    
    private FileTime getLastModifiedTime(File file) throws IOException{
        BasicFileAttributes pmmlByteFileAtt = Files.readAttributes(file.toPath(), BasicFileAttributes.class);
        return pmmlByteFileAtt.lastModifiedTime();
    }
    
    private void fillWithNulls(int numOfNulls, List<?> arr) {

        for (int i = 0; i < numOfNulls; i++) {
            arr.add(null);
        }

    }
}
